#!/bin/sh

# 配置变量
ZONE=''
RECORD=''
EMAIL=''
TOKEN=''
MAX_RETRIES=5 # 最大重试次数
RETRY_INTERVAL_MS=1500 # 重试间隔时间（毫秒）
SUCCESS_STRING='"success":true' # 接口调用成功时响应应包含的字符串

# 代理配置
USE_PROXY=false # 是否使用代理
PROXY_TYPE='' # 代理类型 ('http' 或 'socks5')
PROXY_URL='' # 代理 URL
PROXY_USER='' # 代理用户名（如果不需要认证则留空）
PROXY_PASS='' # 代理密码（如果不需要认证则留空）

# 提取 IPv4 地址的各个八位段
ip_oct1=$(echo "${ip}" | cut -d'.' -f1)
ip_oct2=$(echo "${ip}" | cut -d'.' -f2)
ip_oct3=$(echo "${ip}" | cut -d'.' -f3)
ip_oct4=$(echo "${ip}" | cut -d'.' -f4)

# 计算 IP4P 地址的各个部分
ip_hex_hi=$(printf '%02x%02x' "$ip_oct1" "$ip_oct2")
ip_hex_lo=$(printf '%02x%02x' "$ip_oct3" "$ip_oct4")
port_hex=$(printf '%04x' "${port}")

# 构造 IP4P 地址
ip4p="2001::${port_hex}:${ip_hex_hi}:${ip_hex_lo}"
echo "Generated IP4P Address: $ip4p"

# 初始化重试计数器
retry_count=0

# 重试循环
while true; do
    # 根据 USE_PROXY 和 PROXY_TYPE 变量决定使用的代理选项
    if [ "$USE_PROXY" = true ]; then
        auth=""
        if [ -n "$PROXY_USER" ]; then
            auth="$PROXY_USER"
        fi
        if [ -n "$PROXY_PASS" ]; then
            if [ -n "$auth" ]; then
                auth="$auth:$PROXY_PASS"
            else
                auth="$PROXY_PASS"
            fi
        fi

        case "$PROXY_TYPE" in
        http)
            proxy_option="--proxy http://$auth@$PROXY_URL"
            ;;
        socks5)
            if [ -n "$auth" ]; then
                proxy_option="--socks5 $PROXY_URL --proxy-user $auth"
            else
                proxy_option="--socks5 $PROXY_URL"
            fi
            ;;
        *)
            echo "Unsupported proxy type: $PROXY_TYPE"
            exit 1
            ;;
        esac
    else
        proxy_option=""
    fi

    # 发送请求并获取响应
    response=$(curl -s -X PATCH "https://api.cloudflare.com/client/v4/zones/${ZONE}/dns_records/${RECORD}" \
        -H "X-Auth-Email: ${EMAIL}" \
        -H "Authorization: Bearer ${TOKEN}" \
        -H "Content-Type:application/json" \
        --data "{\"content\":\"${ip4p}\"}" $proxy_option)

    # 检查响应中是否包含成功字符串
    if echo "$response" | grep -q "$SUCCESS_STRING"; then
        echo "Successfully updated DNS record."
        break
    else
        echo "Failed to update DNS record. Response: $response"
        retry_count=$((retry_count + 1))
        if [ "$retry_count" -ge "$MAX_RETRIES" ]; then
            echo "Reached maximum retry limit. Exiting."
            break
        fi
        echo "Retrying... ($retry_count/$MAX_RETRIES)"
        sleep "$(echo "scale=3; $RETRY_INTERVAL_MS / 1000" | bc)"
    fi
done